<?php

class ActionsController extends Controller
{
	public $layout = '//layouts/column2';
	public $defaultAction = 'index';

	/**
	 * @return array action filters
	 */
	public function filters()
	{
		return array(
			'accessControl',
			array(
				'COutputCache + XajaxGetBuildingData',
				'duration' => 60,
			),
			'postOnly + ajaxFireWorker, ajaxGetTrainingMaterials,
				ajaxStartExam, ajaxExamAnswerQuestion, ajaxSubmitExam, ajaxRecruitWorker,
				ajaxMessageMarkRead, markStudentPresence'
		);
	}

	/**
	 * Specifies the access control rules.
	 * This method is used by the 'accessControl' filter.
	 * @return array access control rules
	 */
	public function accessRules()
	{
		return array(
			array('allow', // allow students only
				'actions' => array('ajaxFireWorker', 'markStudentPresence',
					'ajaxGetTrainingMaterials', 'ajaxStartExam', 'ajaxExamAnswerQuestion',
					'ajaxSubmitExam', 'ajaxRecruitWorker', 'ajaxMessageMarkRead',
					'ajaxReportExam'),
				'roles' => array('student')
			),
			array('deny', // deny all users
				'users' => array('*'),
			),
		);
	}

	public function actionAjaxFireWorker()
	{
		header('Content-type: text/json');
		$result = array();

		$transaction = Yii::app()->db->beginTransaction();
		try
		{
			if (!Akcja::isAnyInteriorAllowed('intUP_pracownicy_chem', 'intUP_pracownicy_biol',
				'intUP_pracownicy_mat' ,'intUP_pracownicy_fiz') && !Akcja::isAnyInteriorAllowedAndLessonActive('pracownicy'))
					throw new Exception('Akcja jest zablokowana.');

			$worker_id = filter_input(INPUT_POST, 'worker_id', FILTER_VALIDATE_INT);
			$companyId = Yii::app()->user->id_firmy;

			//znajdź pracownika
			$workerCompany = PracownikFirmy::model()->with('idPracownika0')
				->find('"t"."idPracownika" = :workerId AND "t"."idFirmy" = :companyId AND "t"."dataZwolnienia" IS NULL', array(
					':workerId' => $worker_id,
					':companyId' => $companyId
			));
			/* @var PracownikFirmy $workerCompany */
			if (empty($workerCompany))
				throw new Exception('Nie ma takiego pracownika');

			$stats = $workerCompany->getStatistics();
			if ($stats->inProgress)
				throw new Exception('Pracownik nie może zostać zwolniony ponieważ wykonuje zlecenie.');

			//odejmij punkty pracownika od puktów firmy
			//mozna zmodyfikowac widok PunktyDziedzinyFirmy żeby uwzględnił pracowników
			Yii::app()->db->createCommand('
				update "DziedzinaFirmy" "df"
				set "df"."punkty" = nvl("df"."punkty", 0) - nvl((
					select "dp"."punkty" from "DziedzinaPracownika" "dp"
					where "dp"."idPracownika" = :idPracownika AND "df"."idDziedziny" = "dp"."idDziedziny"
				), 0)
				where "df"."idFirmy" = :idFirmy
			')->execute(array(
				':idFirmy' => $companyId,
				':idPracownika' => $worker_id
			));

			if ($stats->done || (time() - strtotime($workerCompany->dataZatrudnienia)) > 24*3600)
			{
				$workerCompany->dataZwolnienia = new CDbExpression('SYSDATE');
				$workerCompany->save(false);
			}
			else if (!$workerCompany->delete())
				throw new Exception('Nie można zwolnić pracownika.');
			else {
				$company = Firma::model()->findByPk($companyId);
				$points = ParametryGry::getFloat('PUNKTY_ZATR_PRACOWN', 100);
				$company->transferPoints(Dziedzina::MANAGEMENT, -$points);
			}

			$transaction->commit();

			$result['punktyFirmy'] = UserData::getCompanyPoints($companyId);
			$result['rankingFirm'] = UserData::getCompaniesRanking(Yii::app()->user->id_klasy);
		}
		catch (Exception $e)
		{
			$transaction->rollback();
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	/**
	 * POST zatrudnia pracownika
	 * @param $worker_id id Pracownika
	 */
	public function actionAjaxRecruitWorker()
	{
		header('Content-type: text/json');
		$result = array();

		$worker_id = filter_input(INPUT_POST, 'worker_id', FILTER_VALIDATE_INT);
		$payment = filter_input(INPUT_POST, 'payment', FILTER_VALIDATE_FLOAT);
		$id_firmy = Yii::app()->user->id_firmy;

		$transaction = Yii::app()->db->beginTransaction();
		try
		{
			if (!Akcja::isAnyInteriorAllowed('intUP_pracownicy_chem', 'intUP_pracownicy_biol',
				'intUP_pracownicy_mat' ,'intUP_pracownicy_fiz'))
					throw new Exception('Akcja jest zablokowana.');

			//znajdź wolego pracownika
			$pracownik = Pracownik::model()->with('isWorking')->findByPk($worker_id);
			if (empty($pracownik) || $pracownik->isWorking) { /* @var $pracownik Pracownik */
				$result['unavileable'] = 1;
				throw new Exception('Ten pracownik nie jest już dostępny');
			}

			if (!$pracownik->barterPayment($payment))
			{
				$result['minPayment'] = $pracownik->minPayment;
				throw new Exception('Pracownik nie zgadza sie na pracę za podane wynagrodzenie.');
			}

			if (Firma::getFreeSpace($id_firmy) <= 0)
				throw new Exception('W biurze twojej firmy nie ma już wolnych miejsc dla nowych pracowników.');

			//zatrudnij pracownika
			$zatrudnienie = new PracownikFirmy;
			$zatrudnienie->idPracownika = $pracownik->id;
			$zatrudnienie->idFirmy = $id_firmy;
			$zatrudnienie->wynagrodzenie = $payment;
			$zatrudnienie->dataZatrudnienia = new CDbExpression('SYSDATE');
			$zatrudnienie->save(false);

			//przelej punkty pracownika na konto firmy
			//może zmodyfikowac widok PunktyDzidzinyFirmy zeby uwzglednial punkty zatrudnionych pracownikow
			Yii::app()->db->createCommand('
				update "DziedzinaFirmy" "df"
				set "df"."punkty" = nvl("df"."punkty", 0) + nvl((
					select "dp"."punkty" from "DziedzinaPracownika" "dp"
					where "dp"."idPracownika" = :idPracownika AND "df"."idDziedziny" = "dp"."idDziedziny"
				), 0)
				where "df"."idFirmy" = :idFirmy
			')->execute(array(
				':idFirmy' => $id_firmy,
				':idPracownika' => $zatrudnienie->idPracownika
			));

			$company = Firma::model()->findByPk($id_firmy);
			$points = ParametryGry::getFloat('PUNKTY_ZATR_PRACOWN', 100);
			$company->transferPoints(Dziedzina::MANAGEMENT, $points);

			//jak agencja to dodaj fakture płatniczom
			if ($pracownik->idAgencji)
			{
				$result['invoice'] = $pracownik->sell($company);
			}

			$transaction->commit();

			//pobierz aktualną punkację
			$result['punktacja'] = UserData::getCompanyPoints($id_firmy);
		}
		catch (Exception $e)
		{
			$transaction->rollback();
			$result['error'] = $e->getMessage();
		}

		if (Yii::app()->user->getState('bankrut'))
			$result['bankrut'] = 1;

		echo json_encode($result);
		Yii::app()->end();
	}

	/**
	 * Pobiera materiały. Warunkiem jeszt wcześniejsze - jednorazowe - wykupienie kursu.
	 * @param type $szkolenieId Identyfikator szkolenia
	 * @param type $buy Czu automatycznie wykupić
	 */
	public function actionAjaxGetTrainingMaterials()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		$szkolenieId = filter_input(INPUT_POST, 'szkolenieId', FILTER_VALIDATE_INT);
		$buy = filter_input(INPUT_POST, 'buy', FILTER_VALIDATE_BOOLEAN);

		try
		{
			//znajdź szkolenie
			$szkolenie = SzkolenieFirmy::model()->with(array('idSzkolenia0'))->find(array(
				'condition' => '"t"."idFirmy" = :idFirmy AND "t"."idSzkolenia" = :idSzkolenia',
				'params' => array(
					':idSzkolenia' => $szkolenieId,
					':idFirmy' => Yii::app()->user->id_firmy
				),
				'order' => '"t"."dataZakonczenia" DESC, "t"."czySukces" DESC NULLS FIRST'
			));

			//nie ma
			if (empty($szkolenie))
			{
				//automatycznie kup
				if ($buy)
				{
					$company = Firma::model()->findByPk(Yii::app()->user->id_firmy);
					$result = $result + SzkolenieFirmy::wykup($company, $szkolenieId);
				}
				else
				{
					//powidum o tym żeby wykupić
					$result['buy'] = 1;
					throw new Exception('Nie znaleziono szkolenia. Proszę je wykupić.');
				}
			}
			else
			{
				if (strtotime($szkolenie->dataZakonczenia) >= time())
					throw new Exception('Nie można oglądać materiałów podczas trwania egzaminu.');

				//pobierz szkolenie
				$result['szkolenie'] = $szkolenie->attributes + $szkolenie->idSzkolenia0->attributes;
				$result['szkolenie']['czySukces'] = $szkolenie->czySukces;
				$result['szkolenie']['poziomStr'] = $szkolenie->idSzkolenia0->poziomStr;
			}
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		if (Yii::app()->user->getState('bankrut'))
			$result['bankrut'] = 1;

		echo json_encode($result);
		Yii::app()->end();
	}

	/**
	 * Rozpoczyna egzamin
	 * @param type $szkolenieId id szkolenia
	 */
	public function actionAjaxStartExam()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		$szkolenieId = filter_input(INPUT_POST, 'szkolenieId', FILTER_VALIDATE_INT);
		$buy = filter_input(INPUT_POST, 'buy', FILTER_VALIDATE_BOOLEAN);

		try
		{
			$szkolenie = SzkolenieFirmy::model()->with('idSzkolenia0', 'pytaniaEgzaminacyjne.pytanie')->find(array(
				'condition' => '"t"."idSzkolenia" = :idSzkolenia AND "t"."idFirmy" = :idFirmy AND
					"t"."czySukces" IS NULL AND "idSzkolenia0"."czyEgzamin" > 0 AND ("t"."dataRozpoczecia" IS NULL OR
					("t"."dataRozpoczecia" <= to_date(:czas) AND "t"."dataZakonczenia" >= to_date(:czas)))',
				'params' => array(
					':idFirmy' => Yii::app()->user->id_firmy,
					':idSzkolenia' => $szkolenieId,
					':czas' => date('d-m-Y H:i:s', time())
				)
			));
			if (empty($szkolenie))
			{
				if ($buy)
				{
					$firma = Firma::model()->findByPk(Yii::app()->user->id_firmy);
					
					$result = $result + SzkolenieFirmy::wykup($firma, $szkolenieId, true);
					if (isset($result['error']))
						throw new Exception($result['error']);
				}
				else
				{
					$result['buy'] = 1;
					throw new Exception('Musisz wykupić egzamin');
				}
			}
			else
			{
				if (empty($szkolenie->dataRozpoczecia))
				{
					$beginTs = time();
					$endTs = $beginTs + $szkolenie->idSzkolenia0->czasTrwania;
					$szkolenie->dataRozpoczecia = date('d-m-Y H:i:s', $beginTs);
					$szkolenie->dataZakonczenia = date('d-m-Y H:i:s', $endTs);
					if (!$szkolenie->save())
						throw new Exception('Nie można rozpocząć egzaminu.');
				}
				//zwroc dane
				$result['szkolenie'] = $szkolenie->attributes + $szkolenie->idSzkolenia0->attributes;
				unset($result['szkolenie']['tresc']);
				if ($szkolenie->idSzkolenia0->czyEgzamin && is_array($szkolenie->pytaniaEgzaminacyjne))
					foreach ($szkolenie->pytaniaEgzaminacyjne as $pytanie)
						$result['pytania'][] = array(
							'szkolenieId' => $pytanie->idSzkolenieFirmy,
							'pytanieId' => $pytanie->idPytania,
							'wartosc' => empty($pytanie->wartosc) ? '' : $pytanie->wartosc,
							'tresc' => $pytanie->pytanie->tresc,
							'urlObrazka' => $pytanie->pytanie->urlObrazka
						);
				
				$result['szkolenie']['poziomStr'] = $szkolenie->idSzkolenia0->poziomStr;
			}
			$currDate = strtotime(Yii::app()->db->createCommand('select sysdate from dual')->queryScalar()) * 1000;
						
			$result['szkolenie']['startTs'] = strtotime($result['szkolenie']['dataRozpoczecia']) * 1000;
			$result['szkolenie']['endTs'] = strtotime($result['szkolenie']['dataZakonczenia']) * 1000;
			$result['szkolenie']['elasped'] = $currDate - $result['szkolenie']['startTs'];
			$result['szkolenie']['remain'] = $result['szkolenie']['endTs'] - $currDate;
			
//			var_dump($szkolenie->idSzkolenia0->getPoziomStr()); die();

		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		if (Yii::app()->user->getState('bankrut'))
			$result['bankrut'] = 1;

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAjaxExamAnswerQuestion()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		$szkolenieId = filter_input(INPUT_POST, 'szkolenieId', FILTER_VALIDATE_INT);
		$pytanieId = filter_input(INPUT_POST, 'pytanieId', FILTER_VALIDATE_INT);
		$odpowiedz = filter_input(INPUT_POST, 'odpowiedz', FILTER_SANITIZE_STRING);
//		$companyId = Yii::app()->user->id_firmy;
		$now = date('d-m-Y H:i:s');

		try
		{
			$pytanie = SzkolenieOdpowiedziFirmy::model()->find(array(
				'join' => 'left join "SzkolenieFirmy" "sf" ON "t"."idSzkolenieFirmy" = "sf"."id"',
				'condition' => '
					"sf"."idFirmy" = :idFirmy AND
					"t"."idPytania" = :pytanieId AND
					"t"."idSzkolenieFirmy" = :szkolenieId and
					"sf"."czySukces" IS NULL AND
					"sf"."dataRozpoczecia" IS NOT NULL AND
					"sf"."dataZakonczenia" >= sysdate',
				'params' => array(
					':szkolenieId' => $szkolenieId,
					':pytanieId' => $pytanieId,
					':idFirmy' => Yii::app()->user->id_firmy
				)
			));
			if (empty($pytanie))
				throw new Exception('Nie mozna udzielić odpowiedzi dla tego pytania.');
			$pytanie->wartosc = $odpowiedz;
			if (!$pytanie->save())
				throw new Exception('Problem podczas zapisywanie odpowiedzi.');
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAjaxSubmitExam()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		$examId = filter_input(INPUT_POST, 'examId', FILTER_VALIDATE_INT);
		$answers = filter_input(INPUT_POST, 'answers', FILTER_SANITIZE_STRING, FILTER_REQUIRE_ARRAY);
		$companyId = Yii::app()->user->id_firmy;

		$transaction = Yii::app()->db->beginTransaction();
		try
		{
			//pobierz szkolenie (sprawdx czy trwa)
			$szkolenie = SzkolenieFirmy::model()->with(array('pytaniaEgzaminacyjne'))->find(array(
				'condition' => '
					"t"."id" = :idSzkolenia AND
					"t"."idFirmy" = :idFirmy AND
					"t"."czySukces" IS NULL AND
					"t"."dataRozpoczecia" IS NOT NULL AND
					"t"."dataZakonczenia" >= sysdate AND
					"s"."czyEgzamin" > 0',
				'join' => 'left join "Szkolenie" "s" ON "s"."id" = "t"."idSzkolenia"',
				'params' => array(
					':idSzkolenia' => intval($examId),
					':idFirmy' => intval($companyId)
				)
			));
			if (empty($szkolenie))
				throw new Exception('Examinu nie ma badź jest on zakończony');

			//porównaj odpowiedzi i zlicz poprtawne
			$total = 0;
			$valid = 0;
			foreach($szkolenie->pytaniaEgzaminacyjne as $pytEx)
			{
				$result['pe'][] = $pytEx->attributes;
				$odpowiedz = trim(strtolower($answers[$pytEx->idPytania]));
				$foundAnswers = 0;
				foreach ($pytEx->pytanie->odpowiedzi as $odp)
					if ($odpowiedz == trim(strtolower($odp->tresc)))
					{
						$pytEx->idOdpowiedzi = $odp->id;
						$pytEx->wartosc = $odpowiedz;
						if ($pytEx->save())
							$foundAnswers++;
						else
							throw new Exception('Problem podczas zapisywania odpowiedzi');
					}
				if ($foundAnswers)
					$valid++;
				$total++;
			}

			//zapisz stan do bazy
			$progZdawalnosci = (int)ParametryGry::getValue('EGZAMIN_PROG_ZDAWALNOSCI', 3);
			$szkolenie->dataZakonczenia = date('d-m-Y H:i:s', time());
			$szkolenie->czySukces = $valid >= $progZdawalnosci ? 1 : 0;
			if (!$szkolenie->save())
				throw new Exception('Problem podczas zapisywania szkolenia.');
			
			//pobranie poprawnych odpowiedzi i udzielonych odpowiedzi
			$odpowiedzi = SzkolenieOdpowiedziFirmy::model()->with('szkolenieFirmy.idSzkolenia0')->findAll(array(
				'condition' => '"t"."idSzkolenieFirmy" = :idSzkolenieFirmy',
				'params' => array(
					':idSzkolenieFirmy' => $szkolenie->id,
				)
			));
			$i = 0;
			foreach ($odpowiedzi as $odpowiedz)
			{
				$result['answers'][$i]['udzielonaOdpowiedz'] = $odpowiedz->wartosc;

				$arrans = array();
				
				foreach($odpowiedz->pytanie->odpowiedzi as $odp)
					array_push($arrans, $odp->tresc);

				$result['answers'][$i]['poprawneOdpowiedzi'] = implode(', ',$arrans);

				$i++;
			}

			$transaction->commit();

			$result['success'] = $szkolenie->czySukces;
			$result['total'] = $total;
			$result['valid'] = $valid;
			$result['linkToCertificate'] = $this->createUrl('/student/main/certificate', array('id'=>$szkolenie->id));
		}
		catch(Exception $e)
		{
			$transaction->rollback();
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAjaxReportExam()
	{
		header('Content-type: text/json; charset=utf-8');

		$szkolenieId = filter_input(INPUT_POST, 'szkolenieId', FILTER_VALIDATE_INT);
		$szkolenieFirmy = SzkolenieFirmy::model()->find(array(
			'condition' => '
				"id" = :idSzkolenia AND
				"czySukces" = 0 AND
				"zgloszone" IS NULL',
//				"dataRozpoczecia" IS NOT NULL AND
//				"dataZakonczenia" IS NOT NULL',
			'params' => array(
				':idSzkolenia' => intval($szkolenieId),
			)
		));
		if($szkolenieFirmy !== null)
		{
			$result = array();

			$szkolenieFirmy->zgloszone = 1;
			$szkolenieFirmy->save(false);

			echo json_encode($result);
		}
		else
			throw new Exception ('Wystąpił błąd podczas pobierania szkolenia.');

		Yii::app()->end();
	}

	public function actionAjaxMessageMarkRead()
	{
		header('Content-type: text/json');
		$result = array();

		try
		{
			$messageId = filter_input(INPUT_POST, 'messageId', FILTER_VALIDATE_INT);
			$userId = Yii::app()->user->id;

			$action = AkcjaUzytkownika::model()->with(array('akcja'))->findByAttributes(array(
				'id' => $messageId,
				'idUzytkownika' => $userId,
			));
			if (empty($action))
				throw new Exception('Podana akcja nie istnieje.');

			$action->przeczytana = 1;
			if ($action->akcja->czyAutomatyczna)
				$action->dataWykonania = date('d-m-Y h:i:s');
			$action->save(false,array("id","idAkcji","idUzytkownika","dataWykonania","naglowek","idFirmy","dataWyslania","przeczytana"));
			$result['dataWykonania'] = $action->dataWykonania;
			$result['przeczytana'] = $action->przeczytana;
			$result['idakcji'] = $action->idAkcji;
		}
		catch (Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionMarkStudentPresence()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		try
		{
			$studentId = filter_input(INPUT_POST, 'studentId', FILTER_VALIDATE_INT);
			$present = filter_input(INPUT_POST, 'present', FILTER_VALIDATE_INT);

			if (!Yii::app()->user->tura_aktywna)
				throw new Exception('Lekcja została już zakończona.');

			if (!Uzytkownik::isCompanyOwner($studentId, Yii::app()->user->id_firmy))
				throw new Exception('Uczeń nie jest członkiem twojej firmy.');

			$tura = Tura::getTuraForClass(Yii::app()->user->id_klasy);
			if (empty($tura))
				throw new Exception('Nie ma żadnej lekcji.');

			$log = UzytkownikObecnosc::model()->findByAttributes(array(
				'idUzytkownika' => $studentId,
				'idTury' => $tura->id
			));
			if (empty($log))
			{
				$log = new UzytkownikObecnosc;
				$log->idTury = $tura->id;
				$log->idUzytkownika = $studentId;
			}
			$log->obecny = $present ? 1 : 0;
			$log->save();

			$result['present'] = $present;
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}
}
